/*
 TODO: сделать описание теста
*/

#include "l502api.h"
#include "e502api.h"
#include <locale.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <ltimer.h>
#include "dev_funcs.h"
#include "timespec_funcs.h"

#ifndef _WIN32
	#include <pthread.h>
#endif

#if defined(__WIN32__) || defined(__CYGWIN__) || defined(_WIN32)
    #include <windows.h>
#endif

#define ADC_LCH_CNT         1
#define READ_BLOCK_SIZE     4096*66*3
#define READ_TIMEOUT        2000
#define TCP_CONNECTION_TOUT 5000
#define OUT_SIGNAL_SIZE     2000
#define OUT_BLOCK_SIZE      256
#define SEND_TOUT           500
#define VAL_TOLERANCE       0.1

#define DACSTEP_V         1
static double DAC_RANGE_MAX = 10;

// Задержка между изменениями выхода ЦАП, мс
#define ASYNC_SIGNAL_DELAY_MS  1000

// Кол-во миллисекунд в секунде
#define MS_PER_SEC             1000

static int length_tolerance = 100;

// частота АЦП, Гц
static double adc_frequency = 500000;
// длительность одного отсчета АЦП
static double microseconds_per_count;

int flag_verbose = 0;
int flag_exit_on_err = 1;
int flag_write_to_file = 0;

uint32_t err_cnt = 0;


/* номера используемых физических каналов */
static uint32_t f_channels[] = {0, 1};
/* режимы измерения для каналов */
static uint32_t f_ch_modes[] = {X502_LCH_MODE_DIFF, X502_LCH_MODE_DIFF};
/* диапазоны измерения для каналов */
static uint32_t f_ch_ranges[] = {X502_ADC_RANGE_10, X502_ADC_RANGE_10};


int32_t f_setup_params(t_x502_hnd hnd) {
    int32_t err = X502_ERR_OK, i;

    /* устанавливаем параметры логической таблицы АЦП */
    err = X502_SetLChannelCount(hnd, ADC_LCH_CNT);
    for (i=0; (i < ADC_LCH_CNT) && (err == X502_ERR_OK); i++)
        err = X502_SetLChannel(hnd, i, f_channels[i], f_ch_modes[i], f_ch_ranges[i], 0);

    /* устанавливаем частоты ввода для АЦП и цифровых входов */
    if (err == X502_ERR_OK) {
        double dout_freq = adc_frequency;

        err = X502_SetAdcFreq(hnd, &adc_frequency, NULL);
        if (err == X502_ERR_OK) {
            err = X502_SetOutFreq(hnd, &dout_freq);
        }
        if (err == X502_ERR_OK) {
            /* выводим реально установленные значения - те что вернули функции */
            microseconds_per_count = 1000000 / adc_frequency;
            printf("Установлены частоты:\n    Частота сбора АЦП = %0.0f Гц\n    ЦАП = %0.0f Гц\n    µSeconds per cound: %0.2f\n",
                 adc_frequency, dout_freq, microseconds_per_count);

        }
    }

    /* записываем настройки в модуль */
    if (err == X502_ERR_OK)
        err = X502_Configure(hnd, 0);
    return err;
}

typedef enum{
    ERR_STEP_SHIFT = 1,
    ERR_STEP_TOO_LONG
} t_test_errrs;

typedef enum{
    INC,
    DEC
} t_signal_state;

double f_find_avg(double curr_avg, size_t itemcount, double next_val){
    size_t new_cnt = itemcount + 1;
    return ((curr_avg * itemcount) + next_val) / (itemcount + 1);
}

#define LAST_SAMPLES_LEN     64

static double last_samples[LAST_SAMPLES_LEN];
static int last_samples_cnt;

bool is_sample_stable(double sample) {
    static bool last_samples_init = false;

    if (last_samples_cnt == (LAST_SAMPLES_LEN - 1)) {
        last_samples_init = true;
    }

    if (!last_samples_init) {
        return false;
    }

    for (int i = 0; i < last_samples_cnt; i++) {
        if (fabs(last_samples[i] - sample) > VAL_TOLERANCE) {
            return false;
        }
    }
    return true;
}

int f_test_length(const double *data, const size_t size) {

    static size_t total_size = 0;
    static double expected_voltage = 0.0;
    static double step_avg = 0.0;

    static size_t len = 0;
    static size_t old_len = 0;
    static uint32_t tran_count = 0;
    static t_signal_state state = INC;
    int err = 0;
    static bool prev_sample_stable = false;
    static int sinal_unstable = 0;

    const double samples_per_dac_delay = adc_frequency * (ASYNC_SIGNAL_DELAY_MS / MS_PER_SEC);

    if (total_size == 0) {
        old_len = samples_per_dac_delay;
    }

    for (size_t i = 0; i < size; i++) {
        bool sample_stable = is_sample_stable(data[i]);

        if (prev_sample_stable && !sample_stable) {

            if(state == INC){
                expected_voltage += DACSTEP_V;
            }
            if(state == DEC){
                expected_voltage -= DACSTEP_V;
            }

            if(expected_voltage >= DAC_RANGE_MAX) {
                state = DEC;
            }
            if(expected_voltage <= -DAC_RANGE_MAX){
                state = INC;
            }

            int diff = len - old_len;
            printf("Step duration: %0.2f µs; Step diff: %+0.2f µs; Avg. value: %+0.2f\n", len * microseconds_per_count, diff * microseconds_per_count, step_avg);

            old_len = len;
            tran_count++;

            if(abs(diff) > length_tolerance && tran_count > 2){
                printf("ERR: Step diff: %d; Threshold: %d\n", diff, length_tolerance);
                err_cnt++;
                err = ERR_STEP_SHIFT;
            }

        }

        if (sample_stable) {
            step_avg = f_find_avg(step_avg, len, data[i]);
            len++;
            sinal_unstable = 0;
        } else {
            len = 0;
            sinal_unstable++;
        }

        if (len > (samples_per_dac_delay * 2)) {
            printf("ERR: Step size: %ld; Threshold: %.2f\n", len, (samples_per_dac_delay * 2));
            printf("current sample: %.2f\n", data[i]);
            len = 0;
            err = ERR_STEP_TOO_LONG;
        }

        prev_sample_stable = sample_stable;

        last_samples[last_samples_cnt] = data[i];
        last_samples_cnt = (last_samples_cnt + 1) % LAST_SAMPLES_LEN;
    }

    total_size += size;

    printf("total transitions: %d\n", tran_count);
    return err;
}


static t_x502_hnd hnd;

// начальное значение ЦАП, В
static double voltage = 0;

#ifdef _WIN32
DWORD WINAPI async_dac_out(void *arg) {
#else
void * async_dac_out(void *arg) {
#endif

    t_signal_state state = INC;
    t_ltimer timer;
    ltimer_set_ms(&timer, ASYNC_SIGNAL_DELAY_MS);

    while (1) {
        if (ltimer_expired(&timer)) {
            X502_AsyncOutDac(hnd, X502_DAC_CH1, voltage, X502_DAC_FLAGS_VOLT | X502_DAC_FLAGS_CALIBR);
            if(voltage >= DAC_RANGE_MAX) {
                state = DEC;
            }

            if(voltage <= -DAC_RANGE_MAX){
                state = INC;
            }

            if(state == INC) {
                voltage += DACSTEP_V;
            } else {
                voltage -= DACSTEP_V;
            }
            ltimer_reset(&timer);
        }
    }
}

#define OUTPUT_FILE_NAME1   "channel1.csv"

int main(int argc, char **argv) {
    int32_t err = 0;
    uint32_t ver;

    int32_t rcv_size;
    static uint32_t rcv_buf[READ_BLOCK_SIZE];
    static double   adc_data[READ_BLOCK_SIZE];

    int32_t run_time = 35;
    struct timespec start_time;
    struct timespec cur_time;

    t_x502_info dev_info;

    FILE* file1;

    #ifdef _WIN32
        HANDLE thread;
    #else
        pthread_t thread;
    #endif


#ifdef _WIN32
    /* устанавливаем локаль, чтобы можно было выводить по-русски в CP1251 без перевода в OEM */
    setlocale(LC_CTYPE, "");
#endif

    for (int i = 0; i < argc; i++) {
        if (!strcmp(argv[i], "verbose")){
            flag_verbose = 1;
        } else
        if(!strcmp(argv[i], "continue-on-err")){
            flag_exit_on_err = 0;
        } else
        if(!strcmp(argv[i], "run-for")){
            i++;
            run_time = strtol(argv[i], NULL, 10);
        } else
        if(!strcmp(argv[i], "log")){
            flag_write_to_file = 1;
        } else
        if(!strcmp(argv[i], "freq")){
            i++;
            adc_frequency = strtol(argv[i], NULL, 10);
        }else
        if(!strcmp(argv[i], "len-tol")){
            i++;
            length_tolerance = strtol(argv[i], NULL, 10);
        }else
        if(!strcmp(argv[i], "help")){
            fprintf(stderr, "Options:\n");
            fprintf(stderr, "\tverbose\n");
            fprintf(stderr, "\tcontinue-on-err\n");
            fprintf(stderr, "\trun-for <seconds>\n");
            fprintf(stderr, "\tlog\n");
            fprintf(stderr, "\tfreq <Hz>\n");
            fprintf(stderr, "\tlen-tol  <counts>\n");
            exit(0);
        }
    }

    /********** Получение списка устройств и выбор, с каким будем работать ******************/
    hnd = select_dev_from_list(argc, argv, 0);

    err = X502_GetDevInfo(hnd, &dev_info);
    if (err == X502_ERR_OK) {
        if (strcmp(dev_info.name, "E502") == 0 || strcmp(dev_info.name, "L502") == 0) {
            DAC_RANGE_MAX = 5;
        }
    }


    /********************************** Работа с модулем **************************/
    /* если успешно выбрали модуль и установили с ним связь - продолжаем работу */

    err = X502_StreamsDisable(hnd, X502_STREAM_ALL_IN | X502_STREAM_ALL_OUT);

    if (hnd != NULL && err == L502_ERR_OK) {

        uint32_t streams = X502_STREAM_ADC;


        err = X502_StreamsEnable(hnd, streams);
        if (err != X502_ERR_OK) {
            fprintf(stderr, "Ошибка разрешения потоков (%d): %s!", err,
                    X502_GetErrorString(err));
        } else {
            err = f_setup_params(hnd);
            if (err != X502_ERR_OK){
                fprintf(stderr, "Ошибка настройки модуля: %s!\n", X502_GetErrorString(err));
            }
        }

        // выводим начальное значение ЦАП
        X502_AsyncOutDac(hnd, X502_DAC_CH1, voltage, X502_DAC_FLAGS_VOLT | X502_DAC_FLAGS_CALIBR);
        // следующее значение будет выведено из async_dac_out thread
        voltage += DACSTEP_V;

        if (err == X502_ERR_OK) {

            /* разрешаем синхронный ввод. В принципе можно его разрешить только
             * после загрузки первого синала в нашем примере. Но данный пример
             * показывает что мы можем сделать это и до выставления сигнала,
             * например если захотим запустить ввод АЦП до того как будет нужно
             * подать циклический сигнал на вывод */
            if (err == X502_ERR_OK)
                err = X502_StreamsStart(hnd);

            if (err == X502_ERR_OK) {
                int exit = 0;
                if (flag_write_to_file){
                    file1 = fopen(OUTPUT_FILE_NAME1, "w");
                    if (!file1){
                        printf("Error opening "OUTPUT_FILE_NAME1" logfile!\n");
                        return -1;
                    }
                }

                clock_gettime(CLOCK_MONOTONIC, &start_time);

                #ifndef _WIN32
                    pthread_create(&thread, NULL, async_dac_out, NULL);
                #else
                    thread = CreateThread(NULL, 0, async_dac_out, NULL, 0, NULL);
                #endif


                for (int block = 0; (err == X502_ERR_OK); block++) {

                    rcv_size = X502_Recv(hnd, rcv_buf, READ_BLOCK_SIZE, READ_TIMEOUT);
                    if (rcv_size < 0) {
                        err = rcv_size;
                        fprintf(stderr, "Ошибка приема данных: %s\n", X502_GetErrorString(err));
                    }
                    if (rcv_size > 0) {
                        uint32_t adc_size;

                        adc_size = sizeof(adc_data)/sizeof(adc_data[0]);

                        /* обрабатываем принятые данные, распределяя их на данные АЦП и цифровых входов */
                        err = X502_ProcessData(hnd, rcv_buf, rcv_size, X502_PROC_FLAGS_VOLT, adc_data, &adc_size, NULL, NULL);
                        if (err != X502_ERR_OK) {
                            fprintf(stderr, "Ошибка обработки данных: %s\n", X502_GetErrorString(err));
                        } else {
                            uint32_t lch;

                            printf("Блок %3d. Обработано данных АЦП =%d\n", block, adc_size);

                            int ret = f_test_length(adc_data, adc_size);

                            if (ERR_STEP_SHIFT == ret) {
                                printf("Steps shifted %d times!\n", err_cnt);

                            }

                            if(ERR_STEP_TOO_LONG == ret){
                                printf("Step too long!\n");
                            }

                            if (flag_exit_on_err) {
                                err = ret;
                            }

                            if (flag_write_to_file){
                                for (uint32_t i = 0; i < adc_size; i++){
                                    // printf("%6.4f\n", adc_data[i]);
                                    fprintf(file1, "%f\n", adc_data[i]);

                                }
                            }

                            printf("\n");
                            fflush(stdout);
                        }
                    }

                    clock_gettime(CLOCK_MONOTONIC, &cur_time);
                    if(cur_time.tv_sec  - start_time.tv_sec >= run_time) {
                        if(err_cnt > 0) {
                            printf("Finished with errors! ");
                        } else {
                            printf("OK! ");
                        }
                        printf("Ran for: %ld seconds\n",cur_time.tv_sec  - start_time.tv_sec);
                        break;
                    }
                }

                X502_StreamsStop(hnd);
            }
        }

        /* закрываем связь с модулем */
        X502_Close(hnd);
        /* освобождаем описатель */
        X502_Free(hnd);
    }
    return err ? err : err_cnt;
}


