﻿
#include <bitset>
#include <iomanip>
#include <iostream>
#include <math.h>

#include "e502api.h"
#include "l502api.h"
#include "x502api.h"

#include "dev_funcs.h"

#ifdef _WIN32
#define sleep Sleep
#endif

#define MAX_MODULES_CNT 16
/* количество используемых логических каналов */
#define ADC_LCH_CNT  3

/* частота сбора АЦП в Гц*/
#define ADC_FREQ          500000
/* частота кадров (на логический канал). При ADC_FREQ/ADC_LCH_CNT - межкадровой задержки нет */
#define ADC_FRAME_FREQ    (ADC_FREQ/ADC_LCH_CNT)

/* сколько отсчетов считываем за блок */
#define READ_BLOCK_SIZE   4096*200

#define READ_TIMEOUT     2000

/* номера используемых физических каналов */
static uint32_t f_channels[ADC_LCH_CNT] = { 0, 1, 2 };
/* режимы измерения для каналов */
static uint32_t f_ch_modes[ADC_LCH_CNT] = { X502_LCH_MODE_DIFF, X502_LCH_MODE_DIFF, X502_LCH_MODE_DIFF };
/* диапазоны измерения для каналов */
static uint32_t f_ch_ranges[ADC_LCH_CNT] = { X502_ADC_RANGE_10, X502_ADC_RANGE_10, X502_ADC_RANGE_10 };

int32_t f_setup_params(t_x502_hnd hnd) {
    int32_t err = X502_ERR_OK, i;

    /* устанавливаем параметры логической таблицы АЦП */
    err = X502_SetLChannelCount(hnd, ADC_LCH_CNT);
    for (i = 0; (i < ADC_LCH_CNT) && (err == X502_ERR_OK); i++)
        err = X502_SetLChannel(hnd, i, f_channels[i], f_ch_modes[i], f_ch_ranges[i], 0);

    /* устанавливаем частоты ввода для АЦП и цифровых входов */
    if (err == X502_ERR_OK) {
        double f_adc = ADC_FREQ, f_frame = ADC_FRAME_FREQ;
        err = X502_SetAdcFreq(hnd, &f_adc, &f_frame);
        if (err == X502_ERR_OK) {
            /* выводим реально установленные значения - те что вернули функции */
            printf("Установлены частоты:\n    Частота сбора АЦП = %0.0f\n"
                "    Частота на лог. канал = %0.0f\n",
                f_adc, f_frame);
        }
    }

    /* записываем настройки в модуль */
    if (err == X502_ERR_OK)
        err = X502_Configure(hnd, 0);

    return err;
}

#define VAL_TOLERANCE       0.5

int main(int argc, char **argv) {

    setlocale(LC_CTYPE, "");
    t_x502_hnd hnd;

    /********** Получение списка устройств и выбор, с каким будем работать ******************/
    hnd = select_dev_from_list(argc, argv, 0);
    
    /********************************** Работа с модулем **************************/
    /* если успешно выбрали модуль и установили с ним связь - продолжаем работу */
    if (hnd != NULL) {
        int32_t err = X502_ERR_OK;
        err = f_setup_params(hnd);
         /* массив для приема необработанных данных */
        static uint32_t rcv_buf[READ_BLOCK_SIZE];
        static double   adc_data[READ_BLOCK_SIZE];
        double dac1 = 0;
        double dac1_inc = 1;
        double dac2 = 0;
        double dac2_inc = -1;

        while (1) {
            if (dac1 > 9) {
                dac1_inc = -1;
            } else
            if (dac1 < -9) {
                dac1_inc = 1;
            }
            if (dac2 > 9) {
                dac2_inc = -1;
            } else
            if (dac2 < -9) {
                dac2_inc = 1;
            }

            dac1 += dac1_inc;
            dac2 += dac2_inc;

            X502_AsyncOutDac(hnd, X502_DAC_CH1, dac1, X502_DAC_FLAGS_VOLT | X502_DAC_FLAGS_CALIBR);
            X502_AsyncOutDac(hnd, X502_DAC_CH2, dac2, X502_DAC_FLAGS_VOLT | X502_DAC_FLAGS_CALIBR);

            X502_AsyncGetAdcFrame(hnd, X502_PROC_FLAGS_VOLT, READ_TIMEOUT, adc_data);
            std::cout << "ADC1: " <<  adc_data[0] << " ADC2: " << adc_data[1] << std::endl;
            std::cout << "DAC1: " << dac1 << " DAC2: " << dac2 << std::endl;
            if (fabs(adc_data[0] - dac1) > VAL_TOLERANCE) {
                std::cout << "received: " << adc_data[0] << ", expected: " << dac1 << std::endl;
                goto test_failed;
            }
            if (fabs(adc_data[1] - dac2) > VAL_TOLERANCE) {
                std::cout << "received: " << adc_data[1] << ", expected: " << dac2 << std::endl;
                goto test_failed;
            }
        }
test_failed:
        /* закрываем связь с модулем */
        X502_Close(hnd);
        /* освобождаем описатель */
        X502_Free(hnd);
    }

    return 0;
}
