/*
Тестируем монотонность меандра при асинхронном выводе.
dac1 -> adc2
dac2 -> adc1

На 1 канале ЦАП формируется меандр и одновременно проверяются его параметры на 1 канале АЦП
На цифровые выходы асинхронно выдается случайное значение

Флаги:
  async-dac - асинхронно выводить фиксированное значение на канал 2 ЦАП и его проверка на канале 2 АЦП
  expect-zero - ожидание на канале 1 АЦП нуля

  без флагов - на 2 канале ЦАП формируется меандр сдвинутый относительно ЦАП1 и проверяется на 2 канале АЦП

*/

#include "l502api.h"
#include "e502api.h"
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "dev_funcs.h"
#include "timespec_funcs.h"

#define ADC_FREQ            500000
#define ADC_LCH_CNT         2
#define READ_BLOCK_SIZE     4096*66*3
#define READ_TIMEOUT        2000
#define TCP_CONNECTION_TOUT 5000
#define OUT_SIGNAL_SIZE     2000
#define OUT_BLOCK_SIZE      256
#define SEND_TOUT           500


// #define SQUARE1_MIN 3
// #define SQUARE1_MAX 6

// #define SQUARE2_MIN -7
// #define SQUARE2_MAX -2

static double SQUARE1_MIN = 3;
static double SQUARE1_MAX = 6;

static double SQUARE2_MIN = -7;
static double SQUARE2_MAX = -2;


typedef enum {
    ERR_SHIFT = 1,
    ERR_UNEXPECTED_DATA,
    ERR_ASYNC_DAC_INCORRECT_INPUT,
    ERR_TRANSITION_TOO_LONG,
    ERR_VALID_DATA_TIMEOUT,
} errs;

enum {
    SQUARE_SQUARE = 0,
    SIN_SQUARE,
    SINSQUARE_SQUARE
};


#ifndef M_PI
    #define M_PI 3.14159265358979323846
#endif

uint16_t random_hex[] = {
    0x93f8, 0x05f4, 0x9bf6, 0x7ffe, 0x161e, 0xe7ba, 0xf1af, 0x1e0f,
    0x5ec0, 0xc8ff, 0x55e2, 0x9df0, 0x6a6c, 0xa649, 0x1ade, 0x1b24,
    0x343d, 0x7b6b, 0x76b9, 0xc8fc, 0x2e3a, 0x24d9, 0xf189, 0xf1d1,
    0xff71, 0xe23c, 0xa77f, 0x4d8e, 0x2858, 0xa96d, 0xf289, 0x7337,
    0x640b, 0x57e9, 0x39ce, 0xdbd9, 0xb832, 0x315d, 0x441c, 0x863e,
    0xf7b9, 0x8ecf, 0x1010, 0x88da, 0xeb76, 0x9302, 0xc160, 0x7238
};

int flag_verbose = 0;
int flag_exit_on_err = 1;
int flag_ascync_dac = 0;
int flag_write_to_file = 0;
int flag_expect_zero = 0;

uint32_t err_cnt = 0;

typedef double (*f_gen_sig_word)(uint32_t cntr, uint32_t total_size, double amp);
typedef uint32_t (*f_gen_dout_word)(uint32_t cntr, uint32_t total_size);
/* структура, задающая сигналы на 2-х каналах ЦАП и/или1
1на DOUT */
typedef struct {
    uint32_t size; /* количество точек в сигнале */
    double amp_dac1; /* амплитуда сигнала для ЦАП1 */
    f_gen_sig_word gen_func_dac1; /* функция для генерации отсчета для ЦАП1 */
    double amp_dac2; /* амплитуда сигнала для ЦАП2 */
    f_gen_sig_word gen_func_dac2; /* функция для генерации отсчета для ЦАП2 */
    f_gen_dout_word gen_dout; /* функция для генерации слова на цифровой вывод */
} t_sig_struct;

/* номера используемых физических каналов */
static uint32_t f_channels[] = {0, 1};
/* режимы измерения для каналов */
static uint32_t f_ch_modes[] = {X502_LCH_MODE_DIFF, X502_LCH_MODE_DIFF};
/* диапазоны измерения для каналов */
static uint32_t f_ch_ranges[] = {X502_ADC_RANGE_10, X502_ADC_RANGE_10};

static double f_gen_square(uint32_t cntr, uint32_t total_size, double amp) {
    return cntr < total_size/2 ? SQUARE1_MAX : SQUARE1_MIN;
}

static double f_gen_square2(uint32_t cntr, uint32_t total_size, double amp){
    return ((cntr < total_size/4) || (cntr > total_size/4*3)) ? SQUARE2_MIN : SQUARE2_MAX;
}

/* генерация синуса на весь период */
static double f_gen_sin(uint32_t cntr, uint32_t total_size, double amp) {
    return amp*sin(2*M_PI*cntr/total_size);
}

static double f_gen_square_sin(uint32_t cntr, uint32_t total_size, double amp){
    return cntr < total_size/2 ? SQUARE1_MAX : SQUARE1_MAX+amp*sin(2*M_PI*cntr/total_size);
}


/* таблица, задающая сигналы для выдачи на вход */
/* для модуля E16 частота выходного сигнала будет в 2 раза меньше, т.к. частота ЦАП 500КГц против 1МГц у E502 */
static t_sig_struct f_sig_tbl[] = {
    [SQUARE_SQUARE]    = {2000, 2.5, f_gen_square, 5, f_gen_square2, NULL},
    [SIN_SQUARE]       = {5000, 1.5, f_gen_sin, 2.5, f_gen_square, NULL},
    [SINSQUARE_SQUARE] = {5000, 1.5, f_gen_square_sin, 2.5, f_gen_square2, NULL}
};


/* Запись в буфер драйвера блока данных от сигнала
   cntr - номер отчета, соответствующего первому отсчету блока
   size - количество отсчетов на канал (т.е. записывается ch_cnt*size отсчетов)
   sig  - номер сигнала
   ch_cnt - кол-во используемых каналов (это кол-во можно определить по f_sig_tbl[sig],
            но так как мы его уже определили, то передаем сюда, чтобы опять не определять */
static int32_t f_load_block(t_x502_hnd hnd, uint32_t cntr, uint32_t size, uint32_t sig, uint32_t ch_cnt) {
    static double dac_data1[OUT_BLOCK_SIZE], dac_data2[OUT_BLOCK_SIZE];
    static uint32_t dout_data[OUT_BLOCK_SIZE];
    /* массив слов на запись в модуль - содержит смешенные подготовленные данные
       для всех каналов (максимум для 3-х - 2 ЦАП + DOUT) */
    static uint32_t sbuf[3*OUT_BLOCK_SIZE];
    uint32_t i;
    int32_t err = 0;

    /* заполняем массив на вывод */
    for (i=0; i < size; i++) {
        if (f_sig_tbl[sig].gen_func_dac1 != NULL) {
            dac_data1[i] = f_sig_tbl[sig].gen_func_dac1(cntr+i, f_sig_tbl[sig].size, f_sig_tbl[sig].amp_dac1);
        }
        if (f_sig_tbl[sig].gen_func_dac2 != NULL) {
            dac_data2[i] = f_sig_tbl[sig].gen_func_dac2(cntr+i, f_sig_tbl[sig].size, f_sig_tbl[sig].amp_dac2);
        }
        if (f_sig_tbl[sig].gen_dout != NULL) {
            dout_data[i] = f_sig_tbl[sig].gen_dout(cntr+i, f_sig_tbl[sig].size);
        }
    }

    /* Если нужная функция определена, значит мы испоьлзуем этот канал, и
     * подаем на вход сформированный массив. Иначе - канал не используется
     * и передаем на вход NULL */
    err = X502_PrepareData(hnd,
                           f_sig_tbl[sig].gen_func_dac1 ? dac_data1 : NULL,
                           f_sig_tbl[sig].gen_func_dac2 ? dac_data2 : NULL,
                           f_sig_tbl[sig].gen_dout ? dout_data : NULL,
                           size, X502_DAC_FLAGS_VOLT | X502_DAC_FLAGS_CALIBR,
                           sbuf);
    if (err != X502_ERR_OK) {
        fprintf(stderr, "Ошибка подкотовки данных на передачу: %s\n",
                X502_GetErrorString(err));
    } else {
        /* посылаем данные */
        int32_t snd_cnt = X502_Send(hnd, sbuf, size*ch_cnt, SEND_TOUT);
        if (snd_cnt < 0) {
            err = snd_cnt;
            fprintf(stderr, "Ошибка передачи данных: %s\n", X502_GetErrorString(err));
        } else if ((uint32_t)snd_cnt != size*ch_cnt) {
            /* так как мы шлем всегда не больше чем готово, то должны
               всегда передать все */
            fprintf(stderr, "Переданно недостаточно данных: передавали = %d, передано = %d\n",
                    size*ch_cnt, snd_cnt);
            err = X502_ERR_SEND_INSUFFICIENT_WORDS;
        }
    }
    return err;
}

static int32_t f_load_cycle_signal(t_x502_hnd hnd, int sig) {
    int32_t err = 0;
    uint32_t cntr = 0;
    uint32_t ch_cnt=0;

    /* определяем, сколько каналов используем */
    if (f_sig_tbl[sig].gen_func_dac1)
        ch_cnt++;
    if (f_sig_tbl[sig].gen_func_dac2)
        ch_cnt++;
    if (f_sig_tbl[sig].gen_dout)
        ch_cnt++;


    /* задаем размер буфера под все каналы! */
    err = X502_OutCycleLoadStart(hnd, f_sig_tbl[sig].size*ch_cnt);
    if (err != X502_ERR_OK)
        fprintf(stderr, "Ошибка старта загрузки данных: %s!", X502_GetErrorString(err));

    /* в примере показано, что загружать можно поблочно, чтобы не выделять
     * большой размер памяти на весь сигнал в программе. Но можно записать
     * весь сигнал и за один L502_Send() */
    while ((cntr != f_sig_tbl[sig].size) && (err == X502_ERR_OK)) {
        uint32_t block_size = OUT_BLOCK_SIZE;
        /* последний блок может быть меньшего размера, если размер буфера не кратен
         * блоку */
        if (block_size >(f_sig_tbl[sig].size-cntr))
            block_size=f_sig_tbl[sig].size-cntr;

        err = f_load_block(hnd, cntr, block_size, sig, ch_cnt);
        if (!err)
            cntr+=block_size;
    }

    /* делаем активным загруженный сигнал */
    if (err == X502_ERR_OK) {
       err = X502_OutCycleSetup(hnd, X502_OUT_CYCLE_FLAGS_WAIT_DONE);
       if (err != X502_ERR_OK)
            fprintf(stderr, "Ошибка установки циклического сигнала: %s!", X502_GetErrorString(err));
    }
    return err;
}

int32_t f_setup_params(t_x502_hnd hnd) {
    int32_t err = X502_ERR_OK, i;

    /* устанавливаем параметры логической таблицы АЦП */
    err = X502_SetLChannelCount(hnd, ADC_LCH_CNT);
    for (i=0; (i < ADC_LCH_CNT) && (err == X502_ERR_OK); i++)
        err = X502_SetLChannel(hnd, i, f_channels[i], f_ch_modes[i], f_ch_ranges[i], 0);

    /* устанавливаем частоты ввода для АЦП и цифровых входов */
    if (err == X502_ERR_OK) {
        double f_adc = ADC_FREQ;
        double dout_freq = ADC_FREQ;

        err = X502_SetAdcFreq(hnd, &f_adc, NULL);
        if (err == X502_ERR_OK) {
            err = X502_SetOutFreq(hnd, &dout_freq);
        }
        if (err == X502_ERR_OK) {
            /* выводим реально установленные значения - те что вернули функции */
            printf("Установлены частоты:\n    Частота сбора АЦП = %0.0f Гц\n    ЦАП = %0.0f Гц\n", f_adc, dout_freq);
        }
    }

    /* записываем настройки в модуль */
    if (err == X502_ERR_OK)
        err = X502_Configure(hnd, 0);
    return err;
}


enum {
    ADC_CH_1 = 0,
    ADC_CH_2 = 1
};

enum {
    UPPER = 0,
    LOWER = 1
};


#define VAL_TOLERANCE 1.0
#define LEN_TOLERANCE 10
#define ASYNC_VOLTAGE 2
#define MAX_TRANSITION_TIME 10
#define START_TIME_MAX  10

typedef enum {
    LO = 1,
    HI,
    TR  // transition state
} t_ch_state;

double dac_expected_val[2];

int f_test_length(const double *data, const size_t size){

    static size_t total_size = 0;

    static uint32_t len[ADC_LCH_CNT];
    static uint32_t old_len[ADC_LCH_CNT];
    static uint32_t switch_time[ADC_LCH_CNT];

    static t_ch_state prev_ch_state[ADC_LCH_CNT];

    double min[] = {
        [ADC_CH_1] = SQUARE1_MIN - VAL_TOLERANCE,
        [ADC_CH_2] = SQUARE2_MIN - VAL_TOLERANCE,
    };
    double max[] = {
        [ADC_CH_1] = SQUARE1_MAX + VAL_TOLERANCE,
        [ADC_CH_2] = SQUARE2_MAX + VAL_TOLERANCE,
    };

    double threshold_high[ADC_LCH_CNT][2] = {
        [ADC_CH_1] = {SQUARE1_MAX + VAL_TOLERANCE, SQUARE1_MAX - VAL_TOLERANCE},
        [ADC_CH_2] = {SQUARE2_MAX + VAL_TOLERANCE, SQUARE2_MAX - VAL_TOLERANCE},
    };

    double threshold_low[ADC_LCH_CNT][2] = {
        [ADC_CH_1] = {SQUARE1_MIN + VAL_TOLERANCE, SQUARE1_MIN - VAL_TOLERANCE},
        [ADC_CH_2] = {SQUARE2_MIN + VAL_TOLERANCE, SQUARE2_MIN - VAL_TOLERANCE},
    };

    for (size_t i = 0; i < size - ADC_LCH_CNT; i++, total_size++){


        uint8_t channel = i % ADC_LCH_CNT;

        if (total_size < START_TIME_MAX) {
            if (fabs(data[i] - dac_expected_val[channel]) > VAL_TOLERANCE) {
                printf("Count number: %ld, got %6.4fV but expected %6.4f±%6.4fV on CH_%d!\n", total_size, data[i], dac_expected_val[channel], VAL_TOLERANCE, channel+1);
            }
            continue;
        }

        if ((flag_ascync_dac || flag_expect_zero) && (channel == X502_DAC_CH2)){
            if (fabs(data[i] - dac_expected_val[channel]) > VAL_TOLERANCE) {
                printf("Unexpected CH_%d async dac input! Got %6.4fV, expected: %6.4f±%6.4fV\n", channel+1, data[i], dac_expected_val[channel], VAL_TOLERANCE);
                return ERR_ASYNC_DAC_INCORRECT_INPUT;
            }
            continue;
        }

        if((data[i] > threshold_high[channel][UPPER]) || (data[i] < threshold_low[channel][LOWER])){
            printf("Unexpected data on CH_%d! Got %6.4fV, expected min: %6.4fV, max: %6.4fV\n", channel+1, data[i], min[channel], max[channel]);
            printf("total_size = %ld\n", total_size);
            fflush(stdout);
            err_cnt++;
            return ERR_UNEXPECTED_DATA;
        }

        t_ch_state ch_state = TR;

        if (data[i] > threshold_high[channel][LOWER]) {
            ch_state = HI;
        } else
        if (data[i] < threshold_low[channel][UPPER]) {
            ch_state = LO;
        }

        if (ch_state == TR) {
            switch_time[channel]++;
            if (switch_time[channel] > MAX_TRANSITION_TIME) {
                printf("CH_%d, transition too long! Value: %d, Threshold: %d\n", channel+1, switch_time[channel], MAX_TRANSITION_TIME);
                fflush(stdout);
                err_cnt++;
                return ERR_TRANSITION_TOO_LONG;
            }
        }

        // проверяем ширину меандра только при переключении сигнала LO->TR HI->TR или LO->HI
        if (ch_state != prev_ch_state[channel] && prev_ch_state[channel] != TR && old_len[channel]) {
            if ((abs(len[channel] - old_len[channel]) > LEN_TOLERANCE)/* && total_size > 2550*/) {
                printf("CH_%d, Count № %ld, shifted by %d counts!\n", channel+1, total_size, old_len[channel] - len[channel]);
                if(flag_verbose){
                    printf("old_len = %d len = %d\n", old_len[channel], len[channel]);
                    printf("total_size = %ld\n", total_size);
                }
                err_cnt++;
                return ERR_SHIFT;
            }
            old_len[channel] = len[channel];
            len[channel] = 0;

            if(flag_verbose){
                printf("ADC Channel №: %d, Transition №: %d, Length: %d\n", channel+1, switch_time[channel], len[channel]);
            }
        }

        if (data[i] > threshold_high[channel][LOWER] || data[i] < threshold_low[channel][UPPER]) {
            len[channel]++;
            switch_time[channel] = 0;
        }

        prev_ch_state[channel] = ch_state;
    }
    return 0;
}

#define OUTPUT_FILE_NAME1   "channel1.csv"
#define OUTPUT_FILE_NAME2   "channel2.csv"

int main(int argc, char **argv) {
    int32_t err = 0;
    uint32_t ver;
    t_x502_hnd hnd = NULL;

    int32_t rcv_size;
    uint32_t adc_size;
    static uint32_t rcv_buf[READ_BLOCK_SIZE];
    static double   adc_data[READ_BLOCK_SIZE];

    int32_t run_time = 10;
    struct timespec start_time;
    struct timespec cur_time;

    t_x502_info dev_info;

    FILE* file1;
    FILE* file2;

#ifdef _WIN32
    /* устанавливаем локаль, чтобы можно было выводить по-русски в CP1251 без перевода в OEM */
    setlocale(LC_CTYPE, "");
#endif

    for (int i = 0; i < argc; i++) {
        if (!strcmp(argv[i], "verbose")){
            flag_verbose = 1;
        } else
        if(!strcmp(argv[i], "continue-on-err")){
            flag_exit_on_err = 0;
        } else
        if(!strcmp(argv[i], "run-for")){
            i++;
            run_time = strtol(argv[i], NULL, 10);
        } else
        if(!strcmp(argv[i], "async-dac")){
            flag_ascync_dac = 1;
        } else
        if(!strcmp(argv[i], "log")){
            flag_write_to_file = 1;
        } else
        if(!strcmp(argv[i], "expect-zero")){
            flag_expect_zero = 1;
        } else
        if(!strcmp(argv[i], "help")){
            fprintf(stderr, "Options:\n");
            fprintf(stderr, "\tverbose\n");
            fprintf(stderr, "\tcontinue-on-err\n");
            fprintf(stderr, "\trun-for\n");
            fprintf(stderr, "\tasync-dac\n");
            fprintf(stderr, "\texpect-zero\n");
            fprintf(stderr, "\tlog\n");
            exit(0);
        }
    }

    if(flag_expect_zero && flag_ascync_dac) {
        fprintf(stderr, "Options set conflict!\n");
        return -1;
    }

    /********** Получение списка устройств и выбор, с каким будем работать ******************/
    hnd = select_dev_from_list(argc, argv, 0);

    err = X502_GetDevInfo(hnd, &dev_info);
    if (err == X502_ERR_OK) {
        if (strcmp(dev_info.name, "E502") == 0 || strcmp(dev_info.name, "L502") == 0) {
            SQUARE1_MIN = 1;
            SQUARE1_MAX = 5;
            SQUARE2_MIN = -4;
            SQUARE2_MAX = -2;
        }
    }

    dac_expected_val[0] = SQUARE1_MAX;
    dac_expected_val[1] = SQUARE2_MIN;

    /********************************** Работа с модулем **************************/
    /* если успешно выбрали модуль и установили с ним связь - продолжаем работу */

    if (err == X502_ERR_OK) {
        err = X502_StreamsDisable(hnd, X502_STREAM_ALL_IN | X502_STREAM_ALL_OUT);
    }

    if (flag_expect_zero) {
        dac_expected_val[X502_DAC_CH2] = 0;
    }

    if (flag_ascync_dac) {
        dac_expected_val[X502_DAC_CH2] = ASYNC_VOLTAGE;
        X502_AsyncOutDac(hnd, X502_DAC_CH2, ASYNC_VOLTAGE, X502_DAC_FLAGS_VOLT);
    }



    if (hnd != NULL && err == L502_ERR_OK) {

        uint32_t streams = X502_STREAM_ADC | X502_STREAM_DAC1;
        if (!flag_ascync_dac && !flag_expect_zero){
            streams |= X502_STREAM_DAC2;
        }


        err = X502_StreamsEnable(hnd, streams);
        if (err != X502_ERR_OK) {
            fprintf(stderr, "Ошибка разрешения потоков (%d): %s!", err,
                    X502_GetErrorString(err));
        } else {
            err = f_setup_params(hnd);
            if (err != X502_ERR_OK){
                fprintf(stderr, "Ошибка настройки модуля: %s!\n", X502_GetErrorString(err));
            }
        }



        if (err == X502_ERR_OK) {
            /* если хотим, то можем загрузить синхнал до разрешения
             * синхронного ввода. Тогда мы могли бы его запустить, наприемер,
               всесте с АЦП по L502_StreamsStart() */
            err = f_load_cycle_signal(hnd, SQUARE_SQUARE);
            if (err == X502_ERR_OK) {
                do {
                    int buffer_err;
                    uint32_t status;

                    buffer_err = X502_OutGetStatusFlags(hnd, &status);
                    if (buffer_err != X502_ERR_OK) {
                        break;
                    }
                    if (status & X502_OUT_STATUS_FLAG_BUF_WAS_EMPTY) {
                        continue;
                    }
                    /* TODO: на E502 циклический буфер весь перетекает в GD32 и реальное состояние надо узнавать через mailbox
                    if (status & X502_OUT_STATUS_FLAG_BUF_IS_EMPTY) {
                        continue;
                    }*/
                    break;
                } while(1);
            }

            /* разрешаем синхронный ввод. В принципе можно его разрешить только
             * после загрузки первого синала в нашем примере. Но данный пример
             * показывает что мы можем сделать это и до выставления сигнала,
             * например если захотим запустить ввод АЦП до того как будет нужно
             * подать циклический сигнал на вывод */
            if (err == X502_ERR_OK)
                err = X502_StreamsStart(hnd);

            if (err == X502_ERR_OK) {
                int exit = 0;
                if (flag_write_to_file){
                    file1 = fopen(OUTPUT_FILE_NAME1, "w");
                    if (!file1){
                        printf("Error opening "OUTPUT_FILE_NAME1" logfile!\n");
                        return -1;
                    }
                    file2 = fopen(OUTPUT_FILE_NAME2, "w");
                    if (!file2){
                        printf("Error opening "OUTPUT_FILE_NAME2" logfile!\n");
                        return -1;
                    }
                }

                clock_gettime(CLOCK_MONOTONIC, &start_time);

                for (int block = 0; (err == X502_ERR_OK); block++) {

                    if (flag_ascync_dac) {
                        X502_AsyncOutDac(hnd, L502_DAC_CH2, ASYNC_VOLTAGE, X502_DAC_FLAGS_VOLT);
                    }
                    X502_AsyncOutDig(hnd, random_hex[block % sizeof(random_hex)], 0);

                    rcv_size = X502_Recv(hnd, rcv_buf, READ_BLOCK_SIZE, READ_TIMEOUT);
                    if (rcv_size < 0) {
                        err = rcv_size;
                        fprintf(stderr, "Ошибка приема данных: %s\n", X502_GetErrorString(err));
                    }
                    if (rcv_size > 0) {
                        uint32_t first_lch;
                        /* получаем номер логического канала, которому соответствует первый отсчет АЦП в массиве */
                        X502_GetNextExpectedLchNum(hnd, &first_lch);

                        adc_size = sizeof(adc_data)/sizeof(adc_data[0]);

                        /* обрабатываем принятые данные, распределяя их на данные АЦП и цифровых входов */
                        err = X502_ProcessData(hnd, rcv_buf, rcv_size, X502_PROC_FLAGS_VOLT, adc_data, &adc_size, NULL, NULL);
                        if (err != X502_ERR_OK) {
                            fprintf(stderr, "Ошибка обработки данных: %s\n", X502_GetErrorString(err));
                        } else {
                            uint32_t lch;

                            printf("Блок %3d. Обработано данных АЦП =%d\n", block, adc_size);

                            int ret = f_test_length(adc_data, adc_size);
                            if (ret) {
                                if(ret == ERR_SHIFT){
                                    printf("Cycle shifted %d times!\n", err_cnt);
                                }

                                if (flag_exit_on_err) {
                                    err = 1;
                                }
                            }

                            if (flag_write_to_file){
                                for (uint32_t i = 0; i < adc_size; i++){
                                    // printf("%6.4f\n", adc_data[i]);
                                    if(i % 2){
                                        fprintf(file2, "%f\n", adc_data[i]);
                                    } else {
                                        fprintf(file1, "%f\n", adc_data[i]);
                                    }

                                }
                            }

                            printf("\n");
                            fflush(stdout);
                        }
                    }
                    clock_gettime(CLOCK_MONOTONIC, &cur_time);
                    if(cur_time.tv_sec  - start_time.tv_sec >= run_time) {
                        if(err_cnt > 0) {
                            printf("Finished with errors! ");
                        } else {
                            printf("OK! ");
                        }
                        printf("Ran for: %ld seconds\n",cur_time.tv_sec  - start_time.tv_sec);
                        break;
                    }
                }

                X502_StreamsStop(hnd);
            }
        }

        /* закрываем связь с модулем */
        X502_Close(hnd);
        /* освобождаем описатель */
        X502_Free(hnd);
    }
    return err ? err : err_cnt;
}


