#include "x502api_private.h"
#include "x502_eeprom.h"
#include "fast_crc.h"
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "../devs/e502/e502_fpga_regs.h"

#define X502_CHECK_ADDR(hnd, addr, size) ((addr+size)>hnd->info.flash_size ? \
    X502_ERR_FLASH_INVALID_ADDR : (size==0) ? X502_ERR_FLASH_INVALID_SIZE : X502_ERR_OK)


X502_EXPORT(int32_t) X502_FlashRead(t_x502_hnd hnd, uint32_t addr, uint8_t* data,
                                    uint32_t size) {
    int32_t err = X502_CHECK_HND_OPENED(hnd);
    if (err==X502_ERR_OK) {
        err = X502_CHECK_ADDR(hnd, addr, size);
    }
    if (err==X502_ERR_OK) {

        for ( ; (size!=0) && (err==X502_ERR_OK); ) {
            uint32_t rd_size = size;
            if (rd_size > hnd->iface_hnd->flash_rd_size)
                rd_size = hnd->iface_hnd->flash_rd_size;

            err = hnd->iface_hnd->flash_rd(hnd, addr, data, rd_size);
            if (err==X502_ERR_OK) {
                data+=rd_size;
                addr+=rd_size;
                size-=rd_size;
            }
        }
    }
    return err;
}

X502_EXPORT(int32_t) X502_FlashWrite(t_x502_hnd hnd, uint32_t addr,
                                     const uint8_t* data, uint32_t size) {
    int32_t err = X502_CHECK_HND_OPENED(hnd);
    if (err==X502_ERR_OK) {
        err = X502_CHECK_ADDR(hnd, addr, size);
    }
    if (err==X502_ERR_OK) {
        for ( ; (size!=0) && (err==X502_ERR_OK); ) {
            uint32_t wr_size = size;
            if (wr_size > hnd->iface_hnd->flash_wr_size)
                wr_size = hnd->iface_hnd->flash_wr_size;

            err = hnd->iface_hnd->flash_wr(hnd, addr, data, wr_size);
            if (err==X502_ERR_OK) {
                data+=wr_size;
                addr+=wr_size;
                size-=wr_size;
            }
        }
    }
    return err;
}

X502_EXPORT(int32_t) X502_FlashErase(t_x502_hnd hnd, uint32_t addr, uint32_t size) {
    int32_t err = X502_CHECK_HND_OPENED(hnd);
    if (err==X502_ERR_OK) {
        err = X502_CHECK_ADDR(hnd, addr, size);
    }
    if (err==X502_ERR_OK) {
        err = hnd->iface_hnd->flash_erase(hnd, addr, size);
    }
    return err;
}

X502_EXPORT(int32_t) X502_FlashWriteEnable(t_x502_hnd hnd) {
    int32_t err = X502_CHECK_HND_OPENED(hnd);
    if (err==X502_ERR_OK) {
        err = hnd->iface_hnd->flash_set_prot(hnd, X502_EEPROM_PROT_WR_USER, NULL, 0);
    }
    return err;
}

X502_EXPORT(int32_t) X502_FlashWriteDisable(t_x502_hnd hnd) {
    int32_t err = X502_CHECK_HND_OPENED(hnd);
    if (err==X502_ERR_OK) {
        err = hnd->iface_hnd->flash_set_prot(hnd, X502_EEPROM_PROT_ALL, NULL, 0);
    }
    return err;
}


X502_EXPORT(int32_t) X502_FlashSetProtection(t_x502_hnd hnd, uint32_t prot, uint8_t *prot_data, uint32_t prot_data_size) {
    int32_t err = X502_CHECK_HND_OPENED(hnd);
    if (err==X502_ERR_OK) {
        err = hnd->iface_hnd->flash_set_prot(hnd, prot, prot_data, prot_data_size);
    }
    return err;
}

bool x502_is_E16(t_x502_hnd hnd) {
    if (strcmp(hnd->info.name, E16_DEVICE_NAME) == 0) {
        return true;
    } else {
        return false;
    }
}


int x502_check_eeprom_t_x502_descr(t_x502_hnd hnd, uint32_t flags);
int x502_check_eeprom_t_lta37_descr(t_x502_hnd hnd, uint32_t flags);

/** Функция проверяет правильность информации о устройстве, записанной в EEPROM.
    При наличии верной инофрмации, из EEPROM считывается название устройства и
    серийный номер, а так же, при наличии, калибровочные коэффициенты */
int x502_check_eeprom(t_x502_hnd hnd, uint32_t flags) {
    int err;
    if (hnd->set.dev_type == X502_DEV_LTA37) {
        err = x502_check_eeprom_t_lta37_descr(hnd, flags);
    } else {
        err = x502_check_eeprom_t_x502_descr(hnd, flags);
    }
    return err;
}

X502_EXPORT(int32_t) X502_ReloadDevInfo(t_x502_hnd hnd, uint32_t flags) {
    int32_t err = X502_CHECK_HND_OPENED(hnd);
    if ((err==X502_ERR_OK) && (hnd->iface_hnd->reload_dev_info!=NULL))
        err = hnd->iface_hnd->reload_dev_info(hnd);
    if (err==X502_ERR_OK)
        err = x502_check_eeprom(hnd, flags);
    return err;
}
