/* Blackfin on-chip ROM API
 *
 * Copyright 2008 Analog Devices Inc.
 *
 * Licensed under the GPL-2 or later.
 */

#ifndef __BFROM_H__
#define __BFROM_H__

#include <stdint.h>

/* Possible syscontrol action flags */
#define SYSCTRL_READ        0x00000000    /* read registers */
#define SYSCTRL_WRITE       0x00000001    /* write registers */
#define SYSCTRL_SYSRESET    0x00000002    /* perform system reset */
#define SYSCTRL_CORERESET   0x00000004    /* perform core reset */
#define SYSCTRL_SOFTRESET   0x00000006    /* perform core and system reset */
#define SYSCTRL_VRCTL       0x00000010    /* read/write VR_CTL register */
#define SYSCTRL_EXTVOLTAGE  0x00000020    /* VDDINT supplied externally */
#define SYSCTRL_INTVOLTAGE  0x00000000    /* VDDINT generated by on-chip regulator */
#define SYSCTRL_OTPVOLTAGE  0x00000040    /* For Factory Purposes Only */
#define SYSCTRL_PLLCTL      0x00000100    /* read/write PLL_CTL register */
#define SYSCTRL_PLLDIV      0x00000200    /* read/write PLL_DIV register */
#define SYSCTRL_LOCKCNT     0x00000400    /* read/write PLL_LOCKCNT register */
#define SYSCTRL_PLLSTAT     0x00000800    /* read/write PLL_STAT register */
#define SYSCTRL_COLDBOOT    0x40000000    /* when called at cold boot */
#define SYSCTRL_PREBOOT     0x80000000    /* when called from preboot */

typedef struct ADI_SYSCTRL_VALUES {
        uint16_t uwVrCtl;
        uint16_t uwPllCtl;
        uint16_t uwPllDiv;
        uint16_t uwPllLockCnt;
        uint16_t uwPllStat;
} ADI_SYSCTRL_VALUES;

static uint32_t (* const bfrom_SysControl)(uint32_t action_flags, ADI_SYSCTRL_VALUES *power_settings, void *reserved) = (void *)0xEF000038;

/* We need a dedicated function since we need to screw with the stack pointer
 * when resetting.  The on-chip ROM will save/restore registers on the stack
 * when doing a system reset, so the stack cannot be outside of the chip.
 */
__attribute__((__noreturn__))
static inline void bfrom_SoftReset(void *new_stack)
{
        while (1)
                /*
                 * We don't declare the SP as clobbered on purpose, since
                 * it confuses the heck out of the compiler, and this function
                 * never returns
                 */
                __asm__ __volatile__(
                        "sp = %[stack];"
                        "jump (%[bfrom_syscontrol]);"
                        : : [bfrom_syscontrol] "p"(bfrom_SysControl),
                                "q0"(SYSCTRL_SOFTRESET),
                                "q1"(0),
                                "q2"(NULL),
                                [stack] "p"(new_stack)
                );
}

/* OTP Functions */
static uint32_t (* const bfrom_OtpCommand)(uint32_t command, uint32_t value) = (void *)0xEF000018;
static uint32_t (* const bfrom_OtpRead)(uint32_t page, uint32_t flags, uint64_t *page_content) = (void *)0xEF00001A;
static uint32_t (* const bfrom_OtpWrite)(uint32_t page, uint32_t flags, uint64_t *page_content) = (void *)0xEF00001C;

/* otp command: defines for "command" */
#define OTP_INIT                 0x00000001
#define OTP_CLOSE                0x00000002

/* otp read/write: defines for "flags" */
#define OTP_LOWER_HALF           0x00000000 /* select upper/lower 64-bit half (bit 0) */
#define OTP_UPPER_HALF           0x00000001
#define OTP_NO_ECC               0x00000010 /* do not use ECC */
#define OTP_LOCK                 0x00000020 /* sets page protection bit for page */
#define OTP_CHECK_FOR_PREV_WRITE 0x00000080

/* ******************************************************************************************* */
/*                                                                                             */
/*   Return Codes for OtpCommand, OtpRead and OtpWrite() routines                              */
/*                                                                                             */
/* ******************************************************************************************* */


/* Error codes */
#define OTP_SUCCESS          0x0000
/* Error Bits */
#define OTP_MASTER_ERROR     0
#define OTP_WRITE_ERROR      1
#define OTP_READ_ERROR       2
#define OTP_ACC_VIO_ERROR    3
#define OTP_DATA_MULT_ERROR  4
#define OTP_ECC_MULT_ERROR   5
#define OTP_PREV_WR_ERROR    6
#define OTP_SB_DEFECT_ERROR  7

/* Warning Bits */
#define OTP_DATA_SB_WARN     8
#define OTP_ECC_SB_WARN      9


/* ******************************************************************************************* */
/*                                                                                             */
/*   Boot block header's bit fields                                                            */
/*                                                                                             */
/* ******************************************************************************************* */

#define HDRSGN              0xFF000000
#define HDRCHK              0x00FF0000
#define DMACODE             0x0000000F


/* ******************************************************************************************* */
/*                                                                                             */
/*   Boot Flags (part of block header's block code field)                                      */
/*                                                                                             */
/* ******************************************************************************************* */


#define BFLAG_FINAL         0x00008000   /* final block in stream */
#define BFLAG_FIRST         0x00004000   /* first block in stream */
#define BFLAG_INDIRECT      0x00002000   /* load data via intermediate buffer */
#define BFLAG_IGNORE        0x00001000   /* ignore block payload */
#define BFLAG_INIT          0x00000800   /* call initcode routine */
#define BFLAG_CALLBACK      0x00000400   /* call callback routine */
#define BFLAG_QUICKBOOT     0x00000200   /* boot block only when BFLAG_WAKEUP=0 */
#define BFLAG_FILL          0x00000100   /* fill memory with 32-bit argument value */
#define BFLAG_AUX           0x00000020   /* load auxiliary header -- reserved */
#define BFLAG_SAVE          0x00000010   /* save block on power down -- reserved */


/* ******************************************************************************************* */
/*                                                                                             */
/*   Boot Flags (global flags for pFlag word)                                                  */
/*                                                                                             */
/* ******************************************************************************************* */


#define BFLAG_NORESTORE     0x80000000   /* do not restore MMR register when done */
#define BFLAG_RESET         0x40000000   /* issue system reset when done */
#define BFLAG_RETURN        0x20000000   /* issue RTS instead of jumping to EVT1 vector */
#define BFLAG_NEXTDXE       0x10000000   /* parse stream via Next DXE pointer */
#define BFLAG_WAKEUP        0x08000000   /* WURESET bit was a '1', enable quickboot */
#define BFLAG_SLAVE         0x04000000   /* boot mode is a slave mode */
#define BFLAG_PERIPHERAL    0x02000000   /* boot mode is a peripheral mode */
#define BFLAG_NOAUTO        0x01000000   /* skip automatic device detection */
#define BFLAG_ALTERNATE     0x00800000   /* use alternate boot source */
#define BFLAG_FASTREAD      0x00400000   /* use 0xB command in SPI master mode */
#define BFLAG_TYPE          0x00100000   /* device type (number of address bytes) */
   #define BFLAG_TYPE1      0x00000000   /* device type (1 SPI/TWI address bytes, Small Page NAND Flash) */
   #define BFLAG_TYPE2      0x00100000   /* device type (2 SPI/TWI address bytes, Large Page NAND Flash) */
   #define BFLAG_TYPE3      0x00200000   /* device type (3 SPI/TWI address bytes, NAND reserved) */
   #define BFLAG_TYPE4      0x00300000   /* device type (4 SPI/TWI address bytes, NAND reserved) */
#define BFLAG_HDRINDIRECT   0x00080000   /* boot block headers via intermediate buffer */
#define BFLAG_HOOK          0x00040000   /* call hook routine after initialization */
#define BFLAG_TEST          0x00020000   /* factory testing */


/* ******************************************************************************************* */
/*                                                                                             */
/*   Callback Flags                                                                            */
/*                                                                                             */
/* ******************************************************************************************* */


#define CBFLAG_FINAL        0x00000008
#define CBFLAG_FIRST        0x00000004
#define CBFLAG_DIRECT       0x00000001


/* ******************************************************************************************* */
/*                                                                                             */
/*   Boot ROM Jump Table Entries                                                               */
/*                                                                                             */
/* ******************************************************************************************* */


#define BFROM_FINALINIT      0xEF000002
#define BFROM_PDMA           0xEF000004
#define BFROM_MDMA           0xEF000006
#define BFROM_MEMBOOT        0xEF000008
#define BFROM_SPIBOOT        0xEF00000A
#define BFROM_TWIBOOT        0xEF00000C
#define BFROM_OTPBOOT        0xEF00000E
#define BFROM_NANDBOOT       0xEF000010

#define BFROM_SECURE_ENTRY   0xEF000014
#define BFROM_OTP_ECC        0xEF000016
#define BFROM_OTP_COMMAND    0xEF000018
#define BFROM_OTP_READ       0xEF00001A
#define BFROM_OTP_WRITE      0xEF00001C
#define BFROM_ECC_LUTADDR    0xEF00001E

#define BFROM_BOOTKERNEL     0xEF000020
#define BFROM_GETPORT        0xEF000022
#define BFROM_NMI            0xEF000024
#define BFROM_HWERROR        0xEF000026
#define BFROM_EXCEPTION      0xEF000028
#define BFROM_EMUENTRY       0xEF00002A

#define BFROM_CRC32          0xEF000030
#define BFROM_CRC32POLY      0xEF000032
#define BFROM_CRC32CALLBACK  0xEF000034
#define BFROM_CRC32INITCODE  0xEF000036
#define BFROM_SYSCONTROL     0xEF000038

#if defined(__ADSPBF50x__)

    #define BFROM_FLASHPROTECT         0xEF00003A
    #define BFROM_FLASHPROTECTINITCODE 0xEF00003C

#endif

/* ******************************************************************************************* */
/*                                                                                             */
/*   Flags and Argument for BFROM_FLASHPROTECT routine                                         */
/*                                                                                             */
/* ******************************************************************************************* */


#if defined(__ADSPBF50x__)

    #define FLASH_LOCK_CONTROL			0x80000000	/* Controls locking of the upper byte of the FLASH_CONTROL register */
    #define FLASH_PROTECT_ENABLE 		0x20000000	/* Controls the Vpp Flash Protect signal */
    #define FLASH_PROTECT_DISABLE 		0x10000000	/* Controls the Vpp Flash Protect signal */
    #define FLASH_RESET_DISABLE 		0x02000000	/* Controls the Flash Reset signal */
    #define FLASH_RESET_ENABLE			0x01000000	/* Controls the Flash Reset signal */
    #define FLASH_LOCKDOWN_ENABLE 		0x00010000	/* Enables the block lockdown feature */
    #define FLASH_LOCKDOWN_START_BLOCK 	0x000000FF	/* Block number to start the lockdown */
    #define FLASH_LOCKDOWN_NUM_BLOCKS 	0x0000FF00	/* Number of block to lockdown */
    #define FLASH_BLOCK_SHIFT_VAL 		0x00000010	/* shift value for FLASH_LOCKDOWN_START_BLOCK to generate the byte address */
    #define FLASH_MAIN_BLOCK_SIZE 		0x00010000	/* Size in bytes of a main block */

#endif


/* ******************************************************************************************* */
/*                                                                                             */
/*   Boot ROM Data Constants with Bit Fields                                                   */
/*                                                                                             */
/* ******************************************************************************************* */


#define BK_REVISION          0xEF000040
#define BK_ZEROS             0xEF000048
#define BK_ONES              0xEF00004C
#define BK_DATECODE          0xEF000050

#if defined (_LANGUAGE_C)
# define pBK_REVISION         ((const unsigned long *)BK_REVISION)
# define pBK_ZEROS            ((const unsigned long *)BK_ZEROS)
# define pBK_ONES             ((const unsigned long *)BK_ONES)
# define pBK_DATECODE         ((const unsigned long *)BK_DATECODE)
#endif /* _LANGUAGE_C */

#define BK_ID                0xFF000000
#define BK_PROJECT           0x00FF0000
#define BK_VERSION           0x0000FF00
#define BK_UPDATE            0x000000FF

#define BK_YEAR              0xFFFF0000
#define BK_MONTH             0x0000FF00
#define BK_DAY               0x000000FF




/* ******************************************************************************************* */
/*                                                                                             */
/*   Predefined OTP Pages to be used with bfrom_OtpRead()                                      */
/*                                                                                             */
/* ******************************************************************************************* */


#define FPS00                0x0004
#define FPS01                0x0005
#define FPS02                0x0006
#define FPS03                0x0007
#define FPS04                0x0008
#define FPS05                0x0009
#define FPS06                0x000A
#define FPS07                0x000B
#define FPS08                0x000C
#define FPS09                0x000D
#define FPS10                0x000E
#define FPS11                0x000F

#define CPS00                0x0010
#define CPS01                0x0011
#define CPS02                0x0012
#define CPS03                0x0013
#define CPS04                0x0014
#define CPS05                0x0015
#define CPS06                0x0016
#define CPS07                0x0017

#define PBS00                0x0018
#define PBS01                0x0019
#define PBS02                0x001A
#define PBS03                0x001B



/* ******************************************************************************************* */
/*                                                                                             */
/*   Bit Fields in OTP Half Page PBS00L                                                        */
/*                                                                                             */
/* ******************************************************************************************* */


#define OTP_VR_CTL_P             0
#define OTP_VR_CTL_M             0x0000FFFF  /* lower 32-bit word */
#define OTP_PLL_CTL_P            16
#define OTP_PLL_CTL_M            0xFFFF0000  /* lower 32-bit word */
#define OTP_PLL_DIV_P            32
#define OTP_PLL_DIV_M            0x000000FF  /* upper 32-bit word */
#define OTP_SPI_BAUD_P           40
#define OTP_SPI_BAUD_M           0x00000700  /* upper 32-bit word */
#define OTP_SPI_FASTREAD_P       43
#define OTP_SPI_FASTREAD_M       0x00000800  /* upper 32-bit word */
#define OTP_TWI_CLKDIV_P         44
#define OTP_TWI_CLKDIV_M         0x00001000  /* upper 32-bit word */
#define OTP_TWI_PRESCALE_P       45
#define OTP_TWI_PRESCALE_M       0x0000E000  /* upper 32-bit word */
#define OTP_TWI_TYPE_P           48
#define OTP_TWI_TYPE_M           0x00030000  /* upper 32-bit word */
#define OTP_SET_PLL_P            50
#define OTP_SET_PLL_M            0x00040000  /* upper 32-bit word */
#define OTP_SET_VR_P             51
#define OTP_SET_VR_M             0x00080000  /* upper 32-bit word */
#define OTP_RESETOUT_HWAIT_P     52
#define OTP_RESETOUT_HWAIT_M     0x00100000  /* upper 32-bit word */


#if defined(__ADSPBF54x__)

    #define OTP_ALTERNATE_HWAIT_P    53
    #define OTP_ALTERNATE_HWAIT_M    0x00200000  /* upper 32-bit word */

    /* bit 54 reserved */

#elif defined(__ADSPBF52x__) || defined(__ADSPBF51x__)

    /* bits 53 to 54 reserved */

#endif


#define OTP_LOAD_PBS00H_P        55
#define OTP_LOAD_PBS00H_M        0x00800000  /* upper 32-bit word */
#define OTP_LOAD_PBS01L_P        56
#define OTP_LOAD_PBS01L_M        0x01000000  /* upper 32-bit word */
#define OTP_LOAD_PBS01H_P        57
#define OTP_LOAD_PBS01H_M        0x02000000  /* upper 32-bit word */
#define OTP_LOAD_PBS02L_P        58
#define OTP_LOAD_PBS02L_M        0x04000000  /* upper 32-bit word */
#define OTP_LOAD_PBS02H_P        59
#define OTP_LOAD_PBS02H_M        0x08000000  /* upper 32-bit word */
#define OTP_LOAD_PBS03L_P        60
#define OTP_LOAD_PBS03L_M        0x10000000  /* upper 32-bit word */
#define OTP_LOAD_PBS03H_P        61
#define OTP_LOAD_PBS03H_M        0x20000000  /* upper 32-bit word */
#define OTP_INVALID_P            62
#define OTP_INVALID_M            0xC0000000  /* upper 32-bit word */



/* ******************************************************************************************* */
/*                                                                                             */
/*   Bit Fields in OTP Half Page PBS00H                                                        */
/*                                                                                             */
/* ******************************************************************************************* */


#define OTP_EBIU_AMBCTL_P        0
#define OTP_EBIU_AMBCTL_M        0x0000FFFF  /* lower 32-bit word */

#if defined(__ADSPBF54x__)

    #define OTP_EBIU_FCTL_P          16
    #define OTP_EBIU_FCTL_M          0xFFFF0000  /* lower 32-bit word */

    #define OTP_EBIU_MODE_P          32
    #define OTP_EBIU_MODE_M          0x000000FF  /* upper 32-bit word */

#elif defined(__ADSPBF52x__)  || defined(__ADSPBF51x__)

    /* bits 16 to 39 reserved */

#endif


#define OTP_EBIU_AMG_P           40
#define OTP_EBIU_AMG_M           0x00000F00  /* upper 32-bit word */


#if defined(__ADSPBF54x__)

    #define OTP_EBIU_DEVSEQ_P        44
    #define OTP_EBIU_DEVSEQ_M        0x0000F000  /* upper 32-bit word */
    #define OTP_EBIU_DEVCFG_P        48
    #define OTP_EBIU_DEVCFG_M        0xFFFF0000  /* upper 32-bit word */

#elif defined(__ADSPBF52x__)  || defined(__ADSPBF51x__)

    /* bits 16 to 63 reserved */

#endif



/* ******************************************************************************************* */
/*                                                                                             */
/*   Bit Fields in OTP Half Page PBS01L                                                        */
/*                                                                                             */
/* ******************************************************************************************* */


/* bits 0 to 63 reserved */


/* ******************************************************************************************* */
/*                                                                                             */
/*   Bit Fields in OTP Half Page PBS01H                                                        */
/*                                                                                             */
/* ******************************************************************************************* */


#define OTP_BMODE00_DIS_P           0
#define OTP_BMODE00_DIS_M           0x00000001  /* lower 32-bit word */
#define OTP_BMODE01_DIS_P           1
#define OTP_BMODE01_DIS_M           0x00000002  /* lower 32-bit word */
#define OTP_BMODE02_DIS_P           2
#define OTP_BMODE02_DIS_M           0x00000004  /* lower 32-bit word */
#define OTP_BMODE03_DIS_P           3
#define OTP_BMODE03_DIS_M           0x00000008  /* lower 32-bit word */
#define OTP_BMODE04_DIS_P           4
#define OTP_BMODE04_DIS_M           0x00000010  /* lower 32-bit word */
#define OTP_BMODE05_DIS_P           5
#define OTP_BMODE05_DIS_M           0x00000020  /* lower 32-bit word */
#define OTP_BMODE06_DIS_P           6
#define OTP_BMODE06_DIS_M           0x00000040  /* lower 32-bit word */
#define OTP_BMODE07_DIS_P           7
#define OTP_BMODE07_DIS_M           0x00000080  /* lower 32-bit word */
#define OTP_BMODE08_DIS_P           8
#define OTP_BMODE08_DIS_M           0x00000100  /* lower 32-bit word */
#define OTP_BMODE09_DIS_P           9
#define OTP_BMODE09_DIS_M           0x00000200  /* lower 32-bit word */
#define OTP_BMODE10_DIS_P           10
#define OTP_BMODE10_DIS_M           0x00000400  /* lower 32-bit word */
#define OTP_BMODE11_DIS_P           11
#define OTP_BMODE11_DIS_M           0x00000800  /* lower 32-bit word */
#define OTP_BMODE12_DIS_P           12
#define OTP_BMODE12_DIS_M           0x00001000  /* lower 32-bit word */
#define OTP_BMODE13_DIS_P           13
#define OTP_BMODE13_DIS_M           0x00002000  /* lower 32-bit word */
#define OTP_BMODE14_DIS_P           14
#define OTP_BMODE14_DIS_M           0x00004000  /* lower 32-bit word */
#define OTP_BMODE15_DIS_P           15
#define OTP_BMODE15_DIS_M           0x00008000  /* lower 32-bit word */
#define OTP_NFC_CTL_P               16
#define OTP_NFC_CTL_M               0x00FF0000  /* lower 32-bit word */
#define OTP_START_PAGE_P            24
#define OTP_START_PAGE_M            0xFF000000  /* lower 32-bit word */

/* bits 32 to 63 reserved */


/* ******************************************************************************************* */
/*                                                                                             */
/*   Bit Fields in OTP Half Page PBS02L                                                        */
/*                                                                                             */
/* ******************************************************************************************* */


#if defined(__ADSPBF54x__)

    #define OTP_EBIU_DDRCTL0_P          0
    #define OTP_EBIU_DDRCTL0_M          0xFFFFFFFF  /* lower 32-bit word */
    #define OTP_EBIU_DDRCTL1_P          32
    #define OTP_EBIU_DDRCTL1_M          0xFFFFFFFF  /* upper 32-bit word */

#elif defined(__ADSPBF52x__)  || defined(__ADSPBF51x__)

    #define OTP_EBIU_SDGCTL_P           0
    #define OTP_EBIU_SDGCTL_M           0xFFFFFFFF  /* lower 32-bit word */
    #define OTP_EBIU_SDBCTL_P           32
    #define OTP_EBIU_SDBCTL_M           0x0000FFFF  /* upper 32-bit word */
    #define OTP_EBIU_SDRCC_P            48
    #define OTP_EBIU_SDRCC_M            0x0FFF0000  /* upper 32-bit word */

    /* bits 60 to 62 reserved */

    #define OTP_EBIU_POWERON_DUMMY_WRITE_P      63
    #define OTP_EBIU_POWERON_DUMMY_WRITE_M      0x80000000  /* upper 32-bit word */

#endif




/* ******************************************************************************************* */
/*                                                                                             */
/*   Bit Fields in OTP Half Page PBS02H                                                        */
/*                                                                                             */
/* ******************************************************************************************* */


#if defined(__ADSPBF54x__)

    #define OTP_EBIU_DDRCTL2L_P         0
    #define OTP_EBIU_DDRCTL2L_M         0x0000FFFF  /* lower 32-bit word */
    #define OTP_EBIU_DDRCTL3L_P         16
    #define OTP_EBIU_DDRCTL3L_M         0xFFFF0000  /* lower 32-bit word */

    #define OTP_EBIU_DDRQUEL_P          32
    #define OTP_EBIU_DDRQUEL_M          0x0000FFFF  /* upper 32-bit word */

    /* bits 48 to 63 reserved */

#elif defined(__ADSPBF52x__) || defined(__ADSPBF51x__)

   /* bits 0 to 63 reserved */

#endif



/* ******************************************************************************************* */
/*                                                                                             */
/*   Bit Fields in OTP Half Page PBS03L                                                        */
/*                                                                                             */
/* ******************************************************************************************* */


/* bits 0 to 63 reserved */


/* ******************************************************************************************* */
/*                                                                                             */
/*   Bit Fields in OTP Half Page PBS03H                                                        */
/*                                                                                             */
/* ******************************************************************************************* */


/* bits 0 to 63 reserved */


/* ******************************************************************************************* */
/*                                                                                             */
/*   Macros to be used along with _P versions of above bit fields                              */
/*                                                                                             */
/* ******************************************************************************************* */


#define OTP_OFFSET(x) (((x)>>5)<<2)
#define OTP_BITPOS(x) ((x)&0x1F)



/* ******************************************************************************************* */
/*                                                                                             */
/*   Block Cipher Modes of Operation                                                           */
/*                                                                                             */
/* ******************************************************************************************* */


#define BLOCK_CIPHER_MODE_ECB  0
#define BLOCK_CIPHER_MODE_CBC  1
#define BLOCK_CIPHER_MODE_OFB  2
#define BLOCK_CIPHER_MODE_CTR  3




/* ******************************************************************************************* */
/*                                                                                             */
/*   Flags for AesInit() routines                                                              */
/*                                                                                             */
/* ******************************************************************************************* */


#define AES_ENCRYPTION         (0x01)
#define AES_DECRYPTION         (0x02)
#define AES_BOTH               (AES_ENCRYPTION | AES_DECRYPTION)



/* ******************************************************************************************* */
/*                                                                                             */
/*   Return Codes for AES routines                                                             */
/*                                                                                             */
/* ******************************************************************************************* */


#define AES_SUCCESS            0
#define AES_INVALID_KEY_SIZE  -1
#define AES_INVALID_MODE      -2



/* ******************************************************************************************* */
/*                                                                                             */
/*   Size definitions used in SHA-1                                                            */
/*                                                                                             */
/* ******************************************************************************************* */


#define SHA1_SCRATCH_BUFFER_SIZE    184
#define SHA1_HASH_SIZE              20



/* ******************************************************************************************* */
/*                                                                                             */
/*   Instruction ROM Jump Table Entries                                                        */
/*                                                                                             */
/* ******************************************************************************************* */

#if defined(__ADSPBF54x__)

#if defined(__SILICON_REVISION__) && (__SILICON_REVISION__==0x1)

#define BFROM_ARC4_INIT       0xFFA14018
#define BFROM_ARC4_CIPHER     0xFFA1401C

#define BFROM_AES_CIPHER      0xFFA14020
#define BFROM_AES_INV_CIPHER  0xFFA14024
#define BFROM_AES_INIT        0xFFA14028
#define BFROM_AES_KEYEXP      0xFFA1402C
#define BFROM_AES_INV_KEYEXP  0xFFA14030

#define BFROM_SHA1_INIT       0xFFA14860
#define BFROM_SHA1_HASH       0xFFA14B6C

#elif defined(__SILICON_REVISION__) && ((__SILICON_REVISION__==0x2) || (__SILICON_REVISION__==0x3) || (__SILICON_REVISION__==0x4))

#define BFROM_ARC4_INIT       0xFFA14018
#define BFROM_ARC4_CIPHER     0xFFA1401C

#define BFROM_AES_CIPHER      0xFFA14020
#define BFROM_AES_INV_CIPHER  0xFFA14024
#define BFROM_AES_INIT        0xFFA14028
#define BFROM_AES_KEYEXP      0xFFA1402C
#define BFROM_AES_INV_KEYEXP  0xFFA14030

#define BFROM_SHA1_INIT       0xFFA14990
#define BFROM_SHA1_HASH       0xFFA14C9C

#else

#define BFROM_ARC4_INIT       __arc4_init
#define BFROM_ARC4_CIPHER     __arc4_cipher

#define BFROM_AES_CIPHER      __aes_cipher
#define BFROM_AES_INV_CIPHER  __aes_inv_cipher
#define BFROM_AES_INIT        __aes_init
#define BFROM_AES_KEYEXP      __aes_keyexp
#define BFROM_AES_INV_KEYEXP  __aes_inv_keyexp

#define BFROM_SHA1_INIT       __sha1_init
#define BFROM_SHA1_HASH       __sha1_hash

#endif /* __SILICON_REVISION__ */

#elif defined(__ADSPBF523__) || defined(__ADSPBF525__) || defined(__ADSPBF527__)

#if defined(__SILICON_REVISION__) && (__SILICON_REVISION__==0x1)

#define BFROM_SHA1_INIT       0xEF001748
#define BFROM_SHA1_HASH       0xEF001A54

#elif defined(__SILICON_REVISION__) && (__SILICON_REVISION__==0x2)

#define BFROM_SHA1_INIT       0xEF001878
#define BFROM_SHA1_HASH       0xEF001B84

#else

#define BFROM_SHA1_INIT       __sha1_init
#define BFROM_SHA1_HASH       __sha1_hash

#endif /* __SILICON_REVISION__ */

#elif defined(__ADSPBF522__) || defined(__ADSPBF524__) || defined(__ADSPBF526__)

#if defined(__SILICON_REVISION__) && (__SILICON_REVISION__==0x0)

#define BFROM_SHA1_INIT       0xEF001748
#define BFROM_SHA1_HASH       0xEF001A54

#elif defined(__SILICON_REVISION__) && ((__SILICON_REVISION__==0x1) || (__SILICON_REVISION__==0x2))

#define BFROM_SHA1_INIT       0xEF001878
#define BFROM_SHA1_HASH       0xEF001B84

#else

#define BFROM_SHA1_INIT       __sha1_init
#define BFROM_SHA1_HASH       __sha1_hash

#endif /* __SILICON_REVISION__ */

#elif defined(__ADSPBF51x__)

#if defined(__SILICON_REVISION__) && ((__SILICON_REVISION__==0x0) || (__SILICON_REVISION__==0x1) || (__SILICON_REVISION__==0x2))

#define BFROM_SHA1_INIT       0xEF001878
#define BFROM_SHA1_HASH       0xEF001B84

#else

#define BFROM_SHA1_INIT       __sha1_init
#define BFROM_SHA1_HASH       __sha1_hash

#endif /* __SILICON_REVISION__ */

#endif /* __ADSPBF5xx__ */


#endif
