/** @defgroup cmd_process Обработка команд от ПК */

/***************************************************************************//**
    @addtogroup cmd_process
    @{
    @file l502_cmd.h

    Файл содержит описания функций, которые используются для обработки команд
    от ПК.
 ******************************************************************************/


#ifndef L502_BF_CMD_H_
#define L502_BF_CMD_H_

#include <stdint.h>
#include "l502_bf_cmd_defs.h"


/***************************************************************************//**
    @brief Запуск обработки команды от ПК.

    Функция начинает обработку команды от ПК, переданной по HDMA, вызывая нужный
    обработчик по коду команды.
    Вызывается из l502_cmd_check_req() при наличии запроса, но можно запустить
    и из самой прошивки выполнение команды, заполнив поля в g_state.cmd
    и вызвав l502_cmd_start().

    Команда считается завершенной, когда будет вызвана l502_cmd_done(), которая
    может быть вызвана как из l502_cmd_start(), так и позже из другого места.

    @param[in] cmd     Структура, описывающая команду
 ******************************************************************************/
void l502_cmd_start(t_l502_bf_cmd* cmd);

/***************************************************************************//**
    @brief Завершение выполнения команды от ПК

    Пометить комнду как завершенную, записав ее реультат и возвращаемые данные
    в фиксированную область. Если data уже указывает на g_state.cmd.data, т.е.
    данные уже находятся в нужной памяти, то они копироваться не будут

    @param[in] result   Код завершения команды
    @param[in] data     Данные для ПК, возвращаемые командой
    @param[in] size     Размер данных, возвращаемых командой (в 32-битных словах)
*******************************************************************************/
void l502_cmd_done(int32_t result, uint32_t* data, uint32_t size);

/***************************************************************************//**
 * @brief Проверка наличия команды от ПК
 *
 * Функция проверяет, есть ли команда от ПК для которой не была начата обработка
 * (был вызов l502_cmd_set_req(), но еще не был соответствующий вызов
 * l502_cmd_start()).
 *
 * Если есть необработанная команда, то для нее вызывается l502_cmd_start().
*******************************************************************************/
void l502_cmd_check_req(void);

/***************************************************************************//**
 * @brief Установка флага, что есть необработанная команда
 *
 * Функция вызвается из прерывания от HostDMA в случае, если при этой операции
 * HostDMA была записана команда. Функция устанавливает флаг, который будет
 * проверен уже из основной программы с помощью l502_cmd_check_req().
 ******************************************************************************/
void l502_cmd_set_req(void);

#endif

/** @} */
