/** @defgroup global_defs Фиксированная область памяти */

/***************************************************************************//**
    @addtogroup global_defs
    @{
    @file l502_global.h
    Файл содержит определение структур, описывающих область памяти
    BlackFin с фиксированными адресами:
    - Структуры дескрипторов для передачи потоков данных по HostDMA
    - Структру для передачи команд от ПК к BlackFin
*******************************************************************************/

#ifndef L502_GLOBAL_H_
#define L502_GLOBAL_H_

#include <stdint.h>
#include "l502_bf_cmd_defs.h"


/** Версия прошивки BlackFin */
#define L502_BF_FIRM_VERSION  0x00000002

/** Дополнительные возможности, реализованные в прошивке */
#define L502_BF_FIRM_FEATURES (L502_BF_FEATURE_FPGA_REG_ACCESS | L502_BF_FEATURE_OUT_STATUS_FLAGS)


/** Структура дескриптора для передачи потока по HostDMA */ 
typedef struct {
    uint32_t flags; /**< Флаги */
    uint32_t full_size; /**< Полный размер передачи */
    void*    addr;  /**< Начальный адрес данных для передачи */
    uint16_t xcnt;  /**< Кол-во 16-разрядных слов в одном передаваемом блоке */
    uint16_t xmod;  /**< Модификатор указателя */
    uint16_t ycnt;  /**< Не используется */
    uint16_t ymod;  /**< Не используется */
    void*    next_descr; /**< Указатель на следующий дескриптор */
    uint16_t id;        /**< id-номер дескриптора */
    uint16_t valid;    /**< Признак действительности */
    uint32_t udata;     /**< Доп данные */
} t_hdma_stream_descr;


/** Максимальное количество дескрипторов на передачу данных (Bf->ПК) */
#define L502_IN_HDMA_DESCR_CNT  (31)
/** Максимальное количество дескрипторов на прием данных (ПК->Bf) */
#define L502_OUT_HDMA_DESCR_CNT  (31)


/** @brief Описание массива дескрипторов HDMA

    Структура описывает расположение дескрипторов HostDMA для организации
    потока данных между BlackFin и персональным компьютером как на передачу,
    так и на прием */
typedef struct {
    t_hdma_stream_descr in_lb; /**< Последний выполненный дескриптор на ввод */
    t_hdma_stream_descr in[L502_IN_HDMA_DESCR_CNT]; /**< Список дескрипторов на ввод */
    t_hdma_stream_descr out_lb; /**< Последний выполненный дескриптор на вывод */
    t_hdma_stream_descr out[L502_OUT_HDMA_DESCR_CNT]; /**< Список дескрипторов на вывод */
} t_hdma_descr_arr;


/** @brief Описание фиксированной области памяти

    Структура описывает данные в памяти сигнального процессора, которые должны
    быть расположены по фиксированным адресам */
typedef struct {
    t_hdma_descr_arr hdma; /**< 0xFF800000 - дескрипторы для обмена потоком по HDMA*/
    t_l502_bf_cmd cmd;     /**< 0xFF800800 - команда обмена с PC*/
} t_l502_board_state;

/** Структура, содержащая данные фиксированной памяти */
extern volatile t_l502_board_state g_state;

#endif


/** @} */
