/***************************************************************************//**
    @addtogroup user_process
    @{
    @file l502_user_process.c

    Файл содержит простейший вариант реализации пользовательских функций, в котором
    потоки данных передаются без изменения и никакие пользовательские команды не
    обрабатываются. Пользователь может изменить этот файл и добавить здесь свою
    обработку.
*******************************************************************************/

#include "l502_stream.h"
#include "l502_hdma.h"
#include "l502_sport_tx.h"
#include "l502_cmd.h"

#include <stdlib.h>

/***************************************************************************//**
    @brief Обработка принятого массива данных АЦП/DIN.

    Функция вызывается каждый раз, когда обнаружены новые данные от
    АЦП/цифровых входов, пришедшие по SPORT0.

    Функция должна обработать данные и вернуть количество обработанных данных,
    однако эти данные все еще считаются использованными (не могут быть переписаны
    новыми пришедшими данными) до тех пор пока не будет вызвана функция
    stream_in_buf_free()).

    Если функция вернет значение меньше чем size, то функция будут вызванна при
    следующем проходе еще раз с указателем на необработанные данные.

    В текущей реализации просто запускается передача данных по HDMA в ПК

    @param[in] data   Указатель на массив с принятыми данными
    @param[in] size   Количество принятых данных в 32-битных словах
    @return           Функция возвращает количество обработанных данных (от 0 до size).
                      На эти данные не будет вызываться повторно usr_in_proc_data(),
                      но они считаются еще используемыми
*******************************************************************************/
uint32_t usr_in_proc_data(uint32_t* data, uint32_t size) {
    /* если есть свободные дескрипторы на передачу по HDMA - ставим блок на
       передачу. Иначе возвращаем 0, чтобы на обработку этих данных функцию
       вызвали бы позже */
    if (hdma_send_req_rdy()) {
        hdma_send_req_start(data, size, 0);
        return size;
    }
    return 0;
}



/***************************************************************************//**
    @brief Обработка принятого массива с данными ЦАП/DOUT

    Функция вызывается каждый раз, когда обнаружены новые данные, принятые от
    ПК по HDMA.
    Функция должна обработать данные и вернуть количество обработанных данных,
    однако эти данные все еще считаются использованными (не могут быть переписаны
    новыми пришедшими данными) до тех пор пока не будет вызвана функция
    stream_out_buf_free()).

    Если функция вернет значение меньше чем size, то функция будут
    вызвана после еще раз с указателем на необработанные данные.

    В текущей реализации просто запускается передача данных по SPORT
    для вывода на ЦАП/цифровые выходы.

    @param[in] data   Указатель на массив с принятыми данными
    @param[in] size   Количество принятых данных в 32-битных словах
    @return           Функция возвращает количество обработанных данных (от 0 до size).
                      На эти данные не будет вызываться повторно usr_out_proc_data(),
                      но они считаются еще используемыми
 ******************************************************************************/
uint32_t usr_out_proc_data(uint32_t* data, uint32_t size) {

    /* если есть свободные дескрипторы на передачу по HDMA - ставим блок на
       передачу. Иначе возвращаем 0, чтобы на обработку этих данных функцию
       вызвали бы позже */
    if (sport_tx_req_rdy()) {

        /* за один раз можем передать в SPORT не более
           SPORT_TX_REQ_SIZE_MAX слов */
        if (size > SPORT_TX_REQ_SIZE_MAX)
            size = SPORT_TX_REQ_SIZE_MAX;

        sport_tx_start_req(data, size);

        return size;
    }
    return 0;
}






/****************************************************************************//**
    @brief Обработка завершения передачи по HostDMA

    Функция вызывается из обработчика прерывания, когда завершилась передача
    блока данных по HDMA в ПК, поставленного до этого на передачу с
    помощью hdma_send_req_start().

    @param[in] addr    Адрес слова, сразу за последним переданным словом
    @param[in] size    Размер переданных данных в 32-битных словах
   ****************************************************************************/
void hdma_send_done(uint32_t* addr, uint32_t size) {
    stream_in_buf_free(size);
}



/***************************************************************************//**
    @brief Обработка завершения передачи по SPORT

    Функция вызывается из обработчика прерывания при завершении передачи блока данных
    по SPORT'у на цифровые выходы/ЦАП, поставленного до этого на передачу с
    помощью sport_tx_start_req().

    
    @param[in] addr   Адрес слова, сразу за последним переданным словом
    @param[in] size   Размер переданных данных в 32-битных словах */
void sport_tx_done(uint32_t* addr, uint32_t size) {
    stream_out_buf_free(size);
}









/****************************************************************************//**
    @brief Обработка пользовательских команд.

    Функция вызывается при приеме команды от ПК с кодом большим или равным
    #L502_BF_CMD_CODE_USER.

    По завершению обработки необходимо обязательно вызвать
    l502_cmd_done(), указав код завершения команды и
    при необходимости передать данные с результатом

    @param[in] cmd  Структура с описанием принятой команды
 ******************************************************************************/
void usr_cmd_process(t_l502_bf_cmd *cmd) {
    l502_cmd_done(L502_BF_ERR_UNSUP_CMD, NULL, 0);
}

/** @} */








